clear all

* Set Folder to obtain input database

cd ..\Inputs

* Generate empty dataset to fill along with projections

gen x=.
save AKOECD, replace

* Import Projections data provided by Alexander Kopoin

forvalues y=77/95 {
	foreach type in MTB LTB {
		capture {
			import excel PO.xlsx, sheet(`type'`y') cellrange(A1) firstrow clear
			local typel=lower("`type'")
			rename `typel'`y'a year
			gen eo=`y'
			append using AKOECD
			save AKOECD, replace
			noisily disp "Economic Outlook: `y', Type: `type'"
		}
	}
}

foreach var of varlist ??? {
	rename `var' POT`var'
}

* Drop, reshape, reorder

drop x
reshape long POT, i(eo year) j(iso) string
gen ISO=upper(iso)
drop iso
order ISO eo year
sort ISO eo year

* Save temporary dataset and relabel

save AKOECD, replace

label var year "Calendar year"
label var POT  "Potential output in lcus from Kopoin"
label var ISO  "ISO country code"
label var eo   "Number of economic outlook"

* Merge with Vintages database constructed from all the individual EO's

merge 1:1 ISO eo year using ANNUALEOS
sort ISO eo year

label var G           "Output gap given in EO's"
label var P           "Potential output in lcus given in EO's"
label var Y           "Actual output in lcus given in EO's"
label var eo_year     "Year of the Economic Outlook"
label var eo_semester "Semester of the Economic Outlook"

drop _merge

* Replace P with POT when P is unavailable (Kopoin provided more data than
* what was simply available from the EO's)

replace P=POT if P==.
replace P=Y/(G/100+1) if P==.
drop POT G

* Rearrange dataset to make it comparable to IMF data

rename P Plcu
rename Y Ylcu
drop if year<1975 | year>2025
rename year calyear

bys eo: egen year=mean(eo_year)
bys eo: egen semester=mean(eo_semester)

drop eo_year eo_semester
reshape wide Plcu Ylcu, i(ISO eo) j(calyear)
sort ISO year semester eo
order ISO year semester eo

* Save another temporary dataset 

save POECD, replace

* Set Panel

by ISO: gen tt=_n
egen Country=group(ISO)
xtset Country tt
drop if year==.

* Generate Shocks to potential output
* First generate log potential output in lcu

quiet forvalues i=1975(1)2025 {
	gen LPlcu`i'=ln(Plcu`i')
	gen LYlcu`i'=ln(Ylcu`i')
}

* Then generate the change in log potential output from one year to the 
* next (over actual time, not over publications of EO)

forvalues i=1976(1)2025 {
	gen GPlcu`i'=LPlcu`i'-LPlcu`=`i'-1'
	gen GYlcu`i'=LYlcu`i'-LYlcu`=`i'-1'
}

* Then generate the revisions to these growth rates across editions of the WEO's

forvalues i=1976(1)2025 {
	gen DGPlcu`i'=d.GPlcu`i'
}

* Assign the appropriate revision to each time period

gen REVOECD1=.
gen REVOECD5=.
forvalues i=1986(1)2016 {
	replace REVOECD1=DGPlcu`i' if year==`i' & semester==1
	capture drop REVOECD5int1 miss1
	egen    miss1 =rowmiss(DGPlcu`i'-DGPlcu`=`i'+4')
	egen    REVOECD5int1=rowtotal(DGPlcu`i'-DGPlcu`=`i'+4')
	replace REVOECD5int1=. if miss1>0
	replace REVOECD5=REVOECD5int1 if year==`i' & semester==1
	replace REVOECD1=DGPlcu`=`i'+1' if year==`i' & semester==2
	capture drop REVOECD5int2 miss2
	egen    miss2 =rowmiss(DGPlcu`=`i'+1'-DGPlcu`=`i'+5')
	egen    REVOECD5int2=rowtotal(DGPlcu`=`i'+1'-DGPlcu`=`i'+5')
	replace REVOECD5int2=. if miss2>0
	replace REVOECD5=REVOECD5int2 if year==`i' & semester==2
}

* Assign the appropriate growth rates of output to each time period

foreach yp in Y P {
	forvalues r=0(1)4 {
		forvalues j=0(1)1 {
			gen G`yp'OECtp`r'gtm`j'=.
			forvalues i=1986(1)2016 {
				replace G`yp'OECtp`r'gtm`j'=l`j'.G`yp'lcu`=`i'+`r'' if year==`i' & semester==1
				replace G`yp'OECtp`r'gtm`j'=l`j'.G`yp'lcu`=`i'+`r'+1' if year==`i' & semester==2
			}
		}
		forvalues j=1(1)3 {
			gen G`yp'OECtp`r'gtp`j'=.
			forvalues i=1986(1)2016 {
				replace G`yp'OECtp`r'gtp`j'=f`j'.G`yp'lcu`=`i'+`r'' if year==`i' & semester==1
				replace G`yp'OECtp`r'gtp`j'=f`j'.G`yp'lcu`=`i'+`r'+1' if year==`i' & semester==2
			}
		}
	}
	forvalues i=1976(1)2021 {
		gen G`yp'OEClast`i'=.
		replace G`yp'OEClast`i'=G`yp'lcu`i' if year==2016 & semester==2
	}
	forvalues r=0(1)4 {
		gen G`yp'OECtp`r'gT=.
		forvalues i=1976(1)2016 {
			capture drop G`yp'OECtp`r'gTtemp
			by Country: egen G`yp'OECtp`r'gTtemp=mean(G`yp'OEClast`=`i'+`r'+1')
			replace G`yp'OECtp`r'gT=G`yp'OECtp`r'gTtemp if year==`i' & semester==2
		}
	}
	drop G`yp'OECtp?gTtemp G?OEClast*
}

keep ISO year semester REVOECD1 REVOECD5 G?OECtp?g*

* Change folder and save output database

cd ..\Outputs
save OECREV, replace

* Change folder and import database again

cd ..\Inputs
use POECD, clear

* Rename time variables to denote that they correspond to WEO editions

rename year     yearEO
rename semester semesterEO

* Keep only relevant variables

keep ISO yearEO semesterEO Plcu* Ylcu*

* Reshape to have calendar year as an additional variable

reshape long Plcu Ylcu, i(ISO yearEO semesterEO) j(year)

* Generate an identification number for panel and set panel

egen id=group(ISO yearEO semesterEO)
xtset id year

* Generate growth rates of potential and actual

foreach v in P Y {
	gen G`v'lcu=(`v'lcu-L.`v'lcu)/L.`v'lcu*100
}

* drop id for which there is no GYlcu

gen hasGY=1 if GYlcu!=.
by id: egen hasGYinid=sum(hasGY)
drop if hasGYinid==0
drop hasGY*

* Generate HP filtered actual output

local i=1
foreach lambda of numlist 3 6.25 10 15 25 50 100 200 400 800 {
	di `lambda'
	di `i'
	tsfilter hp hpcGY`i' = GYlcu, smooth(`lambda') trend(hpGY`i')
	drop hpc*`i'
	label var hpGY`i' "GYlcu trend component from HP with lambda=`lambda'"
	local ++i
}

* Generate MA filtered actual output   
	   
forvalues terms=1(1)10 {
	di `terms'
	tssmooth ma maGY`terms'=GYlcu, window(`=`terms'-1' 1 0)
	label var maGY`terms' "GYlcu filtered with MA(`terms')"
}

forvalues terms=1(1)10 {
	quiet replace maGY`terms'=. if GYlcu==.
}

* Keep only the appropriate observations

gen     keepmau=0
replace keepmau=1 if year==yearEO+1 & semesterEO==2
replace keepmau=1 if year==yearEO   & semesterEO==1 
keep if keepmau==1

drop year id keepmau Plcu Ylcu
rename yearEO year
rename semesterEO semester
rename *G?* *G?OEC*

* Change folder and save output database

cd ..\Outputs
save OECHPMA, replace

* Change folder and delete temporary databases

cd ..\Inputs
erase AKOECD.dta
erase POECD.dta
